/*
 * File: BouncingBall.java
 * ========================================================
 * A program that simulates a bouncing ball.
 */
import acm.program.*;
import acm.graphics.*;
import acm.util.*;

import java.awt.*;

public class BouncingBall extends GraphicsProgram {
	/* The initial X velocity of the ball. */
	private static final double INITIAL_SPEED = 5.0;
	
	/* How long to delay between frames. */
	private static final double PAUSE_TIME = 1000.0 / 48;
	
	/* The size of the ball. */
	private static final double BALL_SIZE = 50;
	
	/* Gravitational acceleration per frame. */
	private static final double MIN_GRAVITY = 0.3;
	private static final double MAX_GRAVITY = 3.0;
	
	/* Amount that the ball bounces back. */
	private static final double MIN_ELASTICITY = 0.00;
	private static final double MAX_ELASTICITY = 1.00;

	/* Simulates a bouncing ball. */
	public void run() {
		/* Continuously loop, bouncing different balls. */
		while (true) {
			GOval ball = createBall();
			add(ball);
			bounceBallWithRandomness(ball);
			removeAll();
		}
	}
	
	/**
	 * Creates the ball that will be dropped.
	 * 
	 * @return The ball that will be dropped.
	 */
	private GOval createBall() {
		GOval ball = new GOval(0, 0, BALL_SIZE, BALL_SIZE);
		ball.setFilled(true);
		ball.setColor(Color.BLUE);
		return ball;
	}
	
	/**
	 * Bounces the given ball with random gravity and elasticity.
	 *
	 * @param ball The ball that should be bounced.
	 */
	private void bounceBallWithRandomness(GOval ball) {
		/* Choose a random gravity and elasticity. */
		RandomGenerator rgen = RandomGenerator.getInstance();
		double gravity = rgen.nextDouble(MIN_GRAVITY, MAX_GRAVITY);
		double elasticity = rgen.nextDouble(MIN_ELASTICITY, MAX_ELASTICITY);
	
		/* For our sake, show off the gravity and elasticity. */
		displayParameters(gravity, elasticity);		

		/* Actually bounce the ball. */
		bounceBall(ball, gravity, elasticity);
	}
	
	/**
	 * Displays the parameters of the bouncing ball.
	 *
	 * @param gravity The gravity with which the ball accelerates.
	 * @param elasticity The amount which the ball bounces back.
	 */
	private void displayParameters(double gravity, double elasticity) {
		GLabel info = new GLabel("Gravity: " + gravity + ", Elasticity: " + elasticity);
		info.setFont("Arial-32");
		info.setLocation(0, info.getAscent());
		add(info);
	}
	
	/**
	 * Simulates the ball dropping with the given parameters.
	 * 
	 * @param ball The ball to drop.
	 * @param gravity The gravity with which the ball accelerates.
	 * @param elasticity The amount that the ball bounces back.
	 */
	private void bounceBall(GOval ball, double gravity, double elasticity) {	
		/* Track the ball's velocity. */
		double dx = INITIAL_SPEED;
		double dy = 0;
		
		/* Until the ball rolls off-screen, keep bouncing the ball. */
		while (isBallOnScreen(ball)) {
			ball.move(dx, dy);
			
			/* If the ball is above ground, have it accelerate downward. */
			if (isBallAboveGround(ball)) {
				dy += gravity;
			} else {
				/* Bounce the ball back and pull it out of the ground. */
				dy *= -elasticity;
				moveBallOutOfGround(ball);
			}
			
			pause(PAUSE_TIME);
		}
	}
	
	/**
	 * Returns whether the given ball has rolled off the screen.
	 *
	 * @param ball The ball to test.
	 * @return Whether it's still on the screen horizontally.
	 */
	private boolean isBallOnScreen(GOval ball) {
		return ball.getX() < getWidth();
	}
	 
	/**
	 * Given a ball, brings it out of the ground so that it doesn't stick to the floor.
	 *
	 * @ball The ball to pull out of the floor.
	 */
	private void moveBallOutOfGround(GOval ball) {
		ball.move(0, getHeight() - (ball.getY() + ball.getHeight()));
	}
	
	/**
	 * Returns whether the ball is above ground.
	 * 
	 * @param ball The ball to check.
	 * @return Whether it's above ground.
	 */
	private boolean isBallAboveGround(GOval ball) {
		return ball.getY() + ball.getHeight() < getHeight();
	}
}
